<?php

namespace App\Imports;

use Exception;
use App\Models\Hall;
use App\Models\User;
use App\Models\Event;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Maatwebsite\Excel\Concerns\ToModel;
use Illuminate\Support\Facades\Validator;
use Maatwebsite\Excel\Concerns\WithHeadingRow;

class EventImport implements ToModel, WithHeadingRow
{
    protected $errors;

    public function __construct()
    {
        $this->errors = collect();
    }

    public function model(array $row)
    {
        //dd($row);
        $validationErrors = $this->validateRow($row);

        if ($validationErrors) {
            // Convert validation errors to a string
            foreach ($validationErrors as $field => $messages) {
                foreach ($messages as $message) {
                    $this->errors->push("Row {$row['event_name']}: {$message}");
                }
            }
            Log::error('CSV Row Validation Failed', ['row' => $row, 'errors' => $validationErrors]);
            return null; // Skip this row
        }

        $emails = array_map('trim', explode(',', $row['users_email']));
        $users = User::whereIn('email', $emails)->get();
        if ($users->isNotEmpty()) {
            DB::beginTransaction();
            try {
                $event = new Event([
                    'name' => $row['event_name'],
                    'start_at' => $row['start_date'],
                    'end_at' => $row['end_date'],
                    'info' => $row['description'],
                    'num_halls' => $row['number_of_halls'],
                ]);

                // Save the event to get the event ID
                $event->save();
                foreach ($users as $user) {
                    $event->users()->attach($user->id);
                }
                // Create and save the halls
                for ($i = 0; $i < $row['number_of_halls']; $i++) {
                    $hall = new Hall();
                    $hall->event_id = $event->id; // Associate hall with event
                    $hall->name = 'Hall ' . ($i + 1); // Correct naming (increment from 1)
                    $hall->save();
                }
                DB::commit();
                return $event;
            } catch (Exception $ex) {
                DB::rollBack();
                Log::error('Failed to save event or attach users', ['exception' => $ex->getMessage(), 'row' => $row]);
                $this->errors->push("Row {$row['event_name']}: Failed to save event or attach users.");
                return null;
            }
        }

        Log::warning('No users found for given emails', ['emails' => $emails]);
        $this->errors->push("Row {$row['event_name']}: No users found for given emails.");
        return null;
    }

    protected function validateRow(array $row)
    {
        $validator = Validator::make($row, [
            'event_name' => 'required|string|max:255',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_at',
            'description' => 'nullable|string',
            'number_of_halls' => 'required|integer|min:1',
            'users_email' => ['required', 'string', function ($attribute, $value, $fail) {
                $emails = array_map('trim', explode(',', $value));
                foreach ($emails as $email) {
                    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
                        $fail("The {$attribute} contains an invalid email: {$email}");
                    }
                }
            }],
        ]);

        if ($validator->fails()) {
            return $validator->errors()->toArray();
        }

        return null;
    }

    public function getErrors()
    {
        return $this->errors;
    }
}
