<?php

namespace App\Repositories;

use Exception;
use App\Models\Hall;
use App\Models\Event;
use App\Models\BoothRating;
use App\Interfaces\EventInterface;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Http;

class EventRepository implements EventInterface
{
    public  function getAllEvents($search = null,$user_id=null,$paginate=true)
    {
        try {

            $query = Event::with('users');

            // If user_id is not null, filter by user_id
            if ($user_id !== null) {
                $query->whereHas('users', function ($query) use ($user_id) {
                    $query->where('user_id', $user_id);
                });
            }

            // If search is not null, filter by search term
            if ($search !== null) {
                $query->where('name', 'LIKE', '%' . $search . '%');
            }

            // Paginate the results
            if($paginate){
                $events = $query->orderBy("id", "desc")->paginate(10);
            }
            else{
                $events = $query->orderBy("id", "desc")->get();
            }


            return $events;
        } catch (Exception $e) {

            Log::error("Error getting all events" . $e->getMessage());
            throw new \Exception('Error retrieving events', 500);
        }
    }
    public function createEvent($data)
    {

        try {
            //dd($data);
            $event=Event::create($data);
            $event->users()->attach($data['user_ids']);
            //dd($data);
            for($i=0;$i<$data['num_halls'];$i++){
                $hall=new Hall();
                $hall->event_id=$event->id;
                $hall->name='Hall '.$i+1;
                $hall->save();

            }

            return true;
        } catch (Exception $ex) {
           // dd($ex->getMessage(),$ex->getLine());
            Log::error("Error while Creating event" . $ex->getMessage());
            return false;
        }
    }

    public function countEvents($category = null, $user_id = null)
    {
        $query = Event::query();

        //dd($test);
        if ($user_id !== null) {
            $query->whereHas('users', function ($q) use ($user_id) {
                $q->where('user_id', $user_id);
            });
        }

        if ($category == 'completed') {
            $eventsCount =
            $query->whereHas('booths', function ($query) {
                $query->where('result', '!=', 'NOT')
                ->where('draft',false)
                ->whereNotNull('result');
            })->whereDoesntHave('booths', function ($query) {
                $query->where('result', 'NOT')
                ->orWhereNull('result')->orWhere('draft',true);
            })->count();
            //dd($eventsCount);
        } elseif ($category == 'progress') {
            $eventsCount = $query->where(function ($query) {

                $query->whereDoesntHave('booths')

                ->orWhereHas('booths', function ($query) {

                    $query->Where('result', 'NOT')->orWhere('draft',true);
                });
            })->count();
        } elseif ($category != null) {
            $events = $query->whereHas('booths', function ($query) use ($category) {
                $query->where('result', $category)->where('draft',false);
            })->with(['booths' => function ($query) use ($category) {
                $query->where('result', $category);
            }])->get();
            $eventsCount = $events->count();
            $totalSize = $events->reduce(function ($carry, $event) {
                return $carry + $event->booths->sum('size');
            }, 0);
            return ['eventsCount' => $eventsCount, 'size' => $totalSize];
        } else {
            $eventsCount = $query->count();
        }
        //dd($eventsCount);
        return $eventsCount;
    }


    public function getEventData($id)
    {
        $event = Event::with(['booths.standType'=>function($query){
            $query->paginate(10);
        }])->with('users')->findOrFail($id);
        $totalBooths = $event->booths->count();

        $totalSize = $event->booths->sum('size');

        return [
            'event' => $event,
            'total_booths' => $totalBooths,
            'total_size' => $totalSize
        ];

    }
    public function getStats($id){

        $event = Event::with('booths', 'users', 'organizers')->findOrFail($id);
        $totalBooths = $event->booths->count();

        $totalSize = $event->booths->sum('size');




        $completedBoothsCount = $event->booths()
        ->where(function ($query) {
            $query->whereNotNull('result')
                ->where('result', '!=', 'NOT')->where('draft',false);
        })->count();
        // Count booths in progress
        $progressBoothsCount = $event->booths()
        ->where(function ($query) {
            $query->whereNull('result')
                ->orWhere('result', 'NOT')->orWhere('draft',true);
        })->count();
        //dd($event, $progressBoothsCount);
        $goldBoothsCount = $event->booths()
            ->where(function ($query) {
                $query->where('result','GOLD')->where('draft', false);

            })
        ->count();
        $disposableBoothsCount = $event->booths()
            ->where(function ($query) {
                $query->where('result', 'DISPOSABLE')->where('draft', false);
            })
        ->count();
        $goldBoothsSize = $event->booths()
            ->where(function ($query) {
                $query->where('result', 'GOLD')->where('draft', false);
            })
        ->sum('size');
        $disposableBoothsSize = $event->booths()
            ->where(function ($query) {
                $query->where('result', 'DISPOSABLE')->where('draft', false);
            })
        ->sum('size');
        $bronzeBoothsCount = $event->booths()
            ->where(function ($query) {
                $query->where('result', 'BRONZE')->where('draft', false);
            })
        ->count();
        $bronzeBoothsSize = $event->booths()
            ->where(function ($query) {
                $query->where('result', 'BRONZE')->where('draft', false);
            })
        ->sum('size');
        $silverBoothsCount = $event->booths()
            ->where(function ($query) {
                $query->where('result', 'SILVER')->where('draft', false);
            })
        ->count();
        $silverBoothsSize = $event->booths()
            ->where(function ($query) {
                $query->where('result', 'SILVER')->where('draft', false);
            })
        ->sum('size');
        return [
            'event' => $event,
            'total_booths' => $totalBooths,
            'total_size' => $totalSize,
            'completed'=> $completedBoothsCount,
            'pending'=> $progressBoothsCount,
            'goldBoothsCount'=> $goldBoothsCount,
            'bronzeBoothsCount'=> $bronzeBoothsCount,
            'silverBoothsCount'=> $silverBoothsCount,
            'silverBoothsSize'=> $silverBoothsSize,
            'bronzeBoothsSize' => $bronzeBoothsSize,
            'goldBoothsSize' => $goldBoothsSize,
            'disposableBoothsCount'=> $disposableBoothsCount,
            'disposableBoothsSize'=> $disposableBoothsSize
        ];
    }
    public function updateEvent($request, $id)
    {
        try {
            //dd($request);
            $event = Event::whereId($id)->first();
            for ($i = $event->num_halls; $i < $request['num_halls']; $i++) {
                $hall = new Hall();
                $hall->event_id = $event->id;
                $hall->name = 'Hall ' . $i + 1;
                $hall->save();
            }
            $event->name = $request['name'];
            $event->info = $request['info'];
            $event->start_at = $request['start_at'];
            $event->end_at = $request['end_at'];
            $event->num_halls = $request['num_halls'];
            $event->update();
            $event->users()->sync($request['user_ids']);

            return true;
        } catch (Exception $ex) {
            //dd($ex->getMessage(),$ex->getLine());
            Log::error("Error getting Updating event" . $ex->getMessage());
            return false;
        }
    }

    public function deleteEvent($id)
    {
        try {
            $event = Event::find($id);
            $event->delete();
            return true;
        } catch (Exception $ex) {
            Log::error("Error while deleting event" . $ex->getMessage());
            return false;
        }
    }
}
