<?php

namespace App\Imports;

use Exception;
use App\Models\Booth;
use App\Models\Hall;
use App\Models\StandType;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Maatwebsite\Excel\Concerns\ToModel;
use Illuminate\Support\Facades\Validator;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Illuminate\Validation\Rule;

class BoothImport implements ToModel, WithHeadingRow
{
    protected $errors;
    protected $event_id;

    public function __construct($event_id)
    {
        $this->errors = collect();
        $this->event_id= $event_id;
    }
    /**
    * @param array $row
    *
    * @return \Illuminate\Database\Eloquent\Model|null
    */
    public function model(array $row)
    {

        $validationErrors = $this->validateRow($row);

        if ($validationErrors) {
            // Convert validation errors to a string
            foreach ($validationErrors as $field => $messages) {
                foreach ($messages as $message) {
                    $this->errors->push("Stand number {$row['stand_number']}: {$message}");
                }
            }
            Log::error('CSV Row Validation Failed', ['row' => $row, 'errors' => $validationErrors]);
            return null; // Skip this row
        }


            try {
                $booth = new Booth([
                'exhibitor_name' => $row['exhibitor_name'],
                'number' => $row['stand_number'],
                'size' => $row['stand_length']* $row['stand_width'],
                'event_id' => $this->event_id,
                'contractor_name' => $row['contractor_name'],
                'contractor_phone' => $row['contractor_phone'],
                'contractor_email' => $row['contractor_email'],
                'hall_id'=>Hall::where('name', $row['hall_name'])->where('event_id',$this->event_id)->first()->id,
                'stand_type_id'=>StandType::where('name',$row['stand_type'])->first()->id,

                ]);

                // Save the event to get the event ID
                $booth->save();
                // Create and save the halls

                return $booth;
            } catch (Exception $ex) {

                Log::error('Failed to save booth', ['exception' => $ex->getMessage(), 'row' => $row]);
                $this->errors->push("Row with stand number {$row['stand_number']}: Failed to save booth.");
                return null;
            }



    }

    protected function validateRow(array $row)
    {
        $validator = Validator::make($row, [
            'exhibitor_name' => 'required|string|max:255',
            'stand_number' => [
                'required',
                function ($attribute, $value, $fail) use ($row) {
                    if (DB::table('booths')
                        ->where('number', $value)
                        ->where('event_id', $this->event_id)
                        ->exists()
                    ) {
                        $fail('The stand number has already been taken for this event.');
                    }
                },
            ],

            'stand_length'=>'required|numeric',
            'stand_width' => 'required|numeric',
            'contractor_name' => 'nullable|string',
            'contractor_email' => 'nullable|email',
            'contractor_phone' => [
                'nullable',
                'validPhoneNumber', // Custom validation rule
            ],
            'hall_name' => [
                'required',
                'string',
                function ($attribute, $value, $fail) {
                    if (!DB::table('halls')
                        ->where('name', $value)
                        ->where('event_id', $this->event_id)
                        ->exists()) {
                        $fail('The selected hall name is invalid for the given event.');
                    }
                },
            ],
            'stand_type' => [
                'required',
                'string',
                Rule::exists('stand_types', 'name'),
            ],



        ]);

        if ($validator->fails()) {
            return $validator->errors()->toArray();
        }

        return null;
    }

    public function getErrors()
    {
        return $this->errors;
    }
}
