<?php

namespace App\Http\Controllers\Admin;

use Exception;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use App\Interfaces\UserInterface;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;

class UserController extends Controller
{
    private $user;
    function __construct(UserInterface $userInterface)
    {
        $this->middleware('permission:user-list|user-create|user-edit|user-delete', ['only' => ['index', 'show']]);
        $this->middleware('permission:user-create', ['only' => ['create', 'store']]);
        $this->middleware('permission:user-edit', ['only' => ['edit', 'update']]);
        $this->middleware('permission:user-delete', ['only' => ['destroy']]);

        $this->user=$userInterface;
    }
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        try{
            $search = $request->get('search');
            if ($search != null) {
                $users = $this->user->getAllUsers($search);
            }
            else{
                $users = $this->user->getAllUsers();
            }

            $countAdmins=$this->user->countUsers('Admin');
            $countOrganizers=$this->user->countUsers('Organizer');
            $countUsers=$this->user->countUsers('User');
            return view('admin.users.index',['users'=> $users, 'countAdmins'=> $countAdmins, 'countUsers'=> $countUsers, 'countOrganizers'=> $countOrganizers,'search'=>$search]);

        }catch(Exception $ex){
            // Handle the error, if any
            Log::error("Error getting all users: " . $ex->getMessage());
            return response()->json([
                'message' => 'Error getting all users',
                'status' => 500
            ]);
        }



    }
    public function searchUser(Request $request)
    {
        $users = $this->user->getAllUsers($request->search);
        return response()->json($users);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('admin.users.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([

            'email' => ['required', 'email', Rule::unique('users', 'email')->whereNull('deleted_at')],
            'name'=>'required',
            'password'=>'required|min:8',
            'role'=>'required'
        ]);
        $registered=$this->user->createUser($request->all());
        if($registered){
            return redirect()->back()->with('success','User Registered Successfully');
        }
        else{
            return  redirect()->back()->withInput()->with(['failed'=>'Failed to register user! Some server Error occured']);
        }


    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $user= $this->user->getUserData($id);
        //return $user->roles[0]->name;
        return view('admin.users.edit',['user'=>$user]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {

        $request->validate([

            Rule::unique('users', 'email')
            ->whereNull('deleted_at')
            ->ignore($id),
            'name' => 'required',
            //'password' => 'required|min:8',
            'role' => 'required'
        ]);
        $updated=$this->user->updateUser($request->all( ),$id );
        if ($updated) {
            return redirect()->back()->with('success', 'User Updated Successfully');
        } else {

            return  redirect()->back()->withInput()->with(['failed' => 'Failed to update user']);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {

        $deleted = $this->user->deleteUser($id);
        if ($deleted) {
            return redirect()->back()->with('success', 'User Deleted Successfully');
        } else {

            return  redirect()->back()->with(['failed' => 'Failed to delete user']);
        }
    }
}
