<?php

namespace App\Http\Controllers\Admin;

use Exception;
use App\Models\Hall;
use App\Models\User;
use App\Models\Event;
use App\Models\StandType;
use App\Imports\EventImport;
use Illuminate\Http\Request;
use App\Interfaces\EventInterface;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Maatwebsite\Excel\Facades\Excel;

class EventController extends Controller
{
    private $event;
    function __construct(EventInterface $eventInterface)
    {
        $this->middleware('permission:event-list|event-create|event-edit|event-delete', ['only' => ['index']]);
        $this->middleware('permission:event-create', ['only' => ['create', 'store']]);
        $this->middleware('permission:event-edit', ['only' => ['edit', 'update']]);
        //$this->middleware('permission:event-delete', ['only' => ['destroy']]);
        $this->middleware('permission:view-event', ['only' => ['show']]);
        $this->event=$eventInterface;
    }
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $search=$request->get('search');
        if($search!=null){
            $events=$this->event->getAllEvents($search);
        }else{
            $events=$this->event->getAllEvents();
        }
        $count=Event::count();

        return  view('admin.events.index',['events'=>$events,'search'=>$search,'count'=>$count]);
    }

    public function searchEvent(Request $request)
    {
        $user = null;
        //dd($user=Auth::user()->id);
        $role = Auth::user()->getRoleNames()->first();
        //dd("role:",$role);
        if ($role == 'User') {
            $user = Auth::user()->id;
        }
        $users = $this->event->getAllEvents($request->search, $user);
        return response()->json($users);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //$users=User::all();
        $users = User::whereDoesntHave('roles', function ($query) {
            $query->whereIn('name', ['Admin', 'Super Admin']);
        })->get();
        //return $users;
        return view('admin.events.create',['users'=>$users]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {

        $request->validate([
            'name' => 'required',
            'num_halls'=>'required',
            'user_ids' => 'required|array|min:1',
            'user_ids.*' => 'exists:users,id', // Validate each user ID
            'start_at' => 'required|date',
            'end_at' => 'required|date|after_or_equal:start_at',

        ]);
        $start_at = \Carbon\Carbon::parse($request->start_at)->format('m/d/Y');
        $end_at = \Carbon\Carbon::parse($request->end_at)->format('m/d/Y');

        // Assign formatted dates back to request object
        $request->merge([
            'start_at' => $start_at,
            'end_at' => $end_at,
        ]);
        //return $request->all();
        $registered = $this->event->createEvent($request->all());
        if ($registered) {
            return redirect()->back()->with('success', 'Event Registered Successfully');
        } else {
            return  redirect()->back()->withInput()->with(['failed' => 'Failed to Create Event']);
        }

    }


    public function import(Request $request)
    {
        $request->validate([
            'file' => 'required|mimes:csv,txt',
        ]);

        try {
            $import = new EventImport;
            Excel::import($import, $request->file('file'));

            if ($import->getErrors()->isEmpty()) {
                return back()->with('success', 'Events imported successfully.');
            } else {
                return back()->withErrors($import->getErrors()->all())->withInput();
            }
        } catch (Exception $e) {
            Log::error('CSV Import Failed:', ['exception' => $e->getMessage()]);
            return back()->withErrors(['file' => 'CSV import failed. Please try again.'])->withInput();
        }
    }


    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $user = Auth::user();

        // Check if the user has the event with the given ID
        $event = $user->events()->find($id);

        if (!$event && $user->getRoleNames()->first()=='User') {
            throw \Spatie\Permission\Exceptions\UnauthorizedException::forPermissions(['view-event']);
        }
        $eventData=$this->event->getEventData($id);
        $stats=$this->event->getStats($id);
        $event=$eventData['event'];
        //dd($event);
        $total_booths= $eventData['total_booths'];
        $total_size= $eventData['total_size'];
        $halls=Hall::where('event_id',$id)->get();
        $count=$event->count();
        $standTypes=StandType::all();
        //dd(auth()->user()->can('booth-delete'));
        return view('admin.events.view',['stats'=>$stats,'count'=>$count,'halls'=>$halls,'event'=>$event, 'total_booths'=> $total_booths, 'total_size'=> $total_size,'standTypes'=>$standTypes]);

    }

    public function getStats(Request $request){

        $id=$request->input('event_id');
        $eventData = $this->event->getStats($id);
        return $eventData;
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {

        $eventData = $this->event->getEventData($id);
        $event = $eventData['event'];
        $users = User::whereDoesntHave('roles', function ($query) {
            $query->whereIn('name', ['Admin', 'Super Admin']);
        })->get();

        return view('admin.events.edit',['event'=>$event,'users'=>$users]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {

        $request->validate([

            'name' => 'required|string|max:255',
            'info' => 'nullable|string',
            'start_at' => 'required|date',
            'end_at' => 'required|date',
            'user_ids' => 'required|array',
            'user_ids.*' => 'exists:users,id',
        ]);
        $start_at = \Carbon\Carbon::parse($request->start_at)->format('m/d/Y');
        $end_at = \Carbon\Carbon::parse($request->end_at)->format('m/d/Y');

        // Assign formatted dates back to request object
        $request->merge([
            'start_at' => $start_at,
            'end_at' => $end_at,
        ]);
        $updated = $this->event->updateEvent($request->all(), $id);
        if ($updated) {
            return redirect()->back()->with('success', 'Event Updated Successfully');
        } else {

            return  redirect()->back()->withInput()->with(['failed' => 'Failed to update Event']);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {

        if($this->event->deleteEvent($id)){
            return redirect()->route('events.index')->with('success', 'Event Deleted Successfully');
        }
        else{
            return redirect()->back()->with(['failed' => 'Failed to delete Event']);
        }
    }
}
